/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.web.filter;

import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletRequestWrapper;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.servlet.http.HttpServletResponseWrapper;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.URI;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Locale;
import java.util.Set;
import java.util.function.Supplier;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.server.ServletServerHttpRequest;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import org.springframework.util.LinkedCaseInsensitiveMap;
import org.springframework.util.StringUtils;
import org.springframework.web.filter.OncePerRequestFilter;
import org.springframework.web.filter.RelativeRedirectResponseWrapper;
import org.springframework.web.util.ForwardedHeaderUtils;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;
import org.springframework.web.util.UrlPathHelper;

public class ForwardedHeaderFilter
extends OncePerRequestFilter {
    private static final Log logger = LogFactory.getLog(ForwardedHeaderFilter.class);
    private static final Set<String> FORWARDED_HEADER_NAMES = Collections.newSetFromMap(new LinkedCaseInsensitiveMap<Boolean>(10, Locale.ROOT));
    private boolean removeOnly;
    private boolean relativeRedirects;

    public void setRemoveOnly(boolean removeOnly) {
        this.removeOnly = removeOnly;
    }

    public void setRelativeRedirects(boolean relativeRedirects) {
        this.relativeRedirects = relativeRedirects;
    }

    @Override
    protected boolean shouldNotFilter(HttpServletRequest request) {
        for (String headerName : FORWARDED_HEADER_NAMES) {
            if (request.getHeader(headerName) == null) continue;
            return false;
        }
        return true;
    }

    @Override
    protected boolean shouldNotFilterAsyncDispatch() {
        return false;
    }

    @Override
    protected boolean shouldNotFilterErrorDispatch() {
        return false;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        if (this.removeOnly) {
            ForwardedHeaderRemovingRequest wrappedRequest = new ForwardedHeaderRemovingRequest(request);
            filterChain.doFilter(wrappedRequest, response);
        } else {
            ForwardedHeaderExtractingRequest wrappedRequest = null;
            HttpServletResponse wrappedResponse = null;
            try {
                wrappedRequest = new ForwardedHeaderExtractingRequest(request);
                wrappedResponse = this.relativeRedirects ? RelativeRedirectResponseWrapper.wrapIfNecessary(response, HttpStatus.SEE_OTHER) : new ForwardedHeaderExtractingResponse(response, wrappedRequest);
            }
            catch (Throwable ex) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Failed to apply forwarded headers to " + this.formatRequest(request), ex);
                }
                response.sendError(400);
                return;
            }
            filterChain.doFilter(wrappedRequest, wrappedResponse);
        }
    }

    protected String formatRequest(HttpServletRequest request) {
        return "HTTP " + request.getMethod() + " \"" + request.getRequestURI() + "\"";
    }

    @Override
    protected void doFilterNestedErrorDispatch(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        this.doFilterInternal(request, response, filterChain);
    }

    static {
        FORWARDED_HEADER_NAMES.add("Forwarded");
        FORWARDED_HEADER_NAMES.add("X-Forwarded-Host");
        FORWARDED_HEADER_NAMES.add("X-Forwarded-Port");
        FORWARDED_HEADER_NAMES.add("X-Forwarded-Proto");
        FORWARDED_HEADER_NAMES.add("X-Forwarded-Prefix");
        FORWARDED_HEADER_NAMES.add("X-Forwarded-Ssl");
        FORWARDED_HEADER_NAMES.add("X-Forwarded-For");
    }

    private static class ForwardedHeaderRemovingRequest
    extends HttpServletRequestWrapper {
        private final Set<String> headerNames;

        public ForwardedHeaderRemovingRequest(HttpServletRequest request) {
            super(request);
            this.headerNames = ForwardedHeaderRemovingRequest.headerNames(request);
        }

        private static Set<String> headerNames(HttpServletRequest request) {
            Set<String> headerNames = Collections.newSetFromMap(new LinkedCaseInsensitiveMap<Boolean>(Locale.ROOT));
            Enumeration<String> names = request.getHeaderNames();
            while (names.hasMoreElements()) {
                String name = names.nextElement();
                if (FORWARDED_HEADER_NAMES.contains(name)) continue;
                headerNames.add(name);
            }
            return Collections.unmodifiableSet(headerNames);
        }

        @Override
        @Nullable
        public String getHeader(String name) {
            if (FORWARDED_HEADER_NAMES.contains(name)) {
                return null;
            }
            return super.getHeader(name);
        }

        @Override
        public Enumeration<String> getHeaders(String name) {
            if (FORWARDED_HEADER_NAMES.contains(name)) {
                return Collections.emptyEnumeration();
            }
            return super.getHeaders(name);
        }

        @Override
        public Enumeration<String> getHeaderNames() {
            return Collections.enumeration(this.headerNames);
        }
    }

    private static class ForwardedHeaderExtractingRequest
    extends ForwardedHeaderRemovingRequest {
        @Nullable
        private final String scheme;
        private final boolean secure;
        @Nullable
        private final String host;
        private final int port;
        @Nullable
        private final InetSocketAddress remoteAddress;
        private final ForwardedPrefixExtractor forwardedPrefixExtractor;

        ForwardedHeaderExtractingRequest(HttpServletRequest servletRequest) {
            super(servletRequest);
            ServletServerHttpRequest request = new ServletServerHttpRequest(servletRequest);
            URI uri = request.getURI();
            HttpHeaders headers = request.getHeaders();
            UriComponents uriComponents = ForwardedHeaderUtils.adaptFromForwardedHeaders(uri, headers).build();
            int port = uriComponents.getPort();
            this.scheme = uriComponents.getScheme();
            this.secure = "https".equals(this.scheme) || "wss".equals(this.scheme);
            this.host = uriComponents.getHost();
            this.port = port == -1 ? (this.secure ? 443 : 80) : port;
            this.remoteAddress = ForwardedHeaderUtils.parseForwardedFor(uri, headers, request.getRemoteAddress());
            Supplier<HttpServletRequest> requestSupplier = () -> (HttpServletRequest)this.getRequest();
            this.forwardedPrefixExtractor = new ForwardedPrefixExtractor(requestSupplier, this.scheme + "://" + this.host + (String)(port == -1 ? "" : ":" + port));
        }

        @Override
        @Nullable
        public String getScheme() {
            return this.scheme;
        }

        @Override
        @Nullable
        public String getServerName() {
            return this.host;
        }

        @Override
        public int getServerPort() {
            return this.port;
        }

        @Override
        public boolean isSecure() {
            return this.secure;
        }

        @Override
        public String getContextPath() {
            return this.forwardedPrefixExtractor.getContextPath();
        }

        @Override
        public String getRequestURI() {
            return this.forwardedPrefixExtractor.getRequestUri();
        }

        @Override
        public StringBuffer getRequestURL() {
            return this.forwardedPrefixExtractor.getRequestUrl();
        }

        @Override
        @Nullable
        public String getRemoteHost() {
            return this.remoteAddress != null ? this.remoteAddress.getHostString() : super.getRemoteHost();
        }

        @Override
        @Nullable
        public String getRemoteAddr() {
            return this.remoteAddress != null ? this.remoteAddress.getHostString() : super.getRemoteAddr();
        }

        @Override
        public int getRemotePort() {
            return this.remoteAddress != null ? this.remoteAddress.getPort() : super.getRemotePort();
        }

        @Override
        @Nullable
        public Object getAttribute(String name) {
            if (name.equals("jakarta.servlet.error.request_uri")) {
                return this.forwardedPrefixExtractor.getErrorRequestUri();
            }
            return super.getAttribute(name);
        }
    }

    private static class ForwardedHeaderExtractingResponse
    extends HttpServletResponseWrapper {
        private static final String FOLDER_SEPARATOR = "/";
        private final HttpServletRequest request;

        ForwardedHeaderExtractingResponse(HttpServletResponse response, HttpServletRequest request) {
            super(response);
            this.request = request;
        }

        @Override
        public void sendRedirect(String location) throws IOException {
            UriComponentsBuilder builder = UriComponentsBuilder.fromUriString(location);
            UriComponents uriComponents = builder.build();
            if (uriComponents.getScheme() != null) {
                super.sendRedirect(location);
                return;
            }
            if (location.startsWith("//")) {
                String scheme = this.request.getScheme();
                super.sendRedirect(builder.scheme(scheme).toUriString());
                return;
            }
            String path = uriComponents.getPath();
            if (path != null) {
                path = path.startsWith(FOLDER_SEPARATOR) ? path : StringUtils.applyRelativePath(this.request.getRequestURI(), path);
            }
            ServletServerHttpRequest httpRequest = new ServletServerHttpRequest(this.request);
            URI uri = httpRequest.getURI();
            HttpHeaders headers = httpRequest.getHeaders();
            String result = ForwardedHeaderUtils.adaptFromForwardedHeaders(uri, headers).replacePath(path).replaceQuery(uriComponents.getQuery()).fragment(uriComponents.getFragment()).build().normalize().toUriString();
            super.sendRedirect(result);
        }
    }

    private static class ErrorPathRequest
    extends HttpServletRequestWrapper {
        ErrorPathRequest(ServletRequest request) {
            super((HttpServletRequest)request);
        }

        @Override
        public String getRequestURI() {
            String requestUri = (String)this.getAttribute("jakarta.servlet.error.request_uri");
            Assert.isTrue(requestUri != null, "Expected ERROR requestUri attribute");
            return requestUri;
        }
    }

    private static class ForwardedPrefixExtractor {
        private final Supplier<HttpServletRequest> delegate;
        private final String baseUrl;
        private String actualRequestUri;
        @Nullable
        private final String forwardedPrefix;
        @Nullable
        private String requestUri;
        private String requestUrl;

        public ForwardedPrefixExtractor(Supplier<HttpServletRequest> delegate, String baseUrl) {
            this.delegate = delegate;
            this.baseUrl = baseUrl;
            this.actualRequestUri = delegate.get().getRequestURI();
            this.forwardedPrefix = ForwardedPrefixExtractor.initForwardedPrefix(delegate.get());
            this.requestUri = this.initRequestUri();
            this.requestUrl = this.initRequestUrl();
        }

        @Nullable
        private static String initForwardedPrefix(HttpServletRequest request) {
            String result = null;
            Enumeration<String> names = request.getHeaderNames();
            while (names.hasMoreElements()) {
                String name = names.nextElement();
                if (!"X-Forwarded-Prefix".equalsIgnoreCase(name)) continue;
                result = request.getHeader(name);
            }
            if (result != null) {
                String[] rawPrefixes;
                StringBuilder prefix = new StringBuilder(result.length());
                for (String rawPrefix : rawPrefixes = StringUtils.tokenizeToStringArray(result, ",")) {
                    int endIndex;
                    for (endIndex = rawPrefix.length(); endIndex > 0 && rawPrefix.charAt(endIndex - 1) == '/'; --endIndex) {
                    }
                    prefix.append(endIndex != rawPrefix.length() ? rawPrefix.substring(0, endIndex) : rawPrefix);
                }
                return prefix.toString();
            }
            return null;
        }

        @Nullable
        private String initRequestUri() {
            if (this.forwardedPrefix != null) {
                return this.forwardedPrefix + UrlPathHelper.rawPathInstance.getPathWithinApplication(this.delegate.get());
            }
            return null;
        }

        private String initRequestUrl() {
            return this.baseUrl + (this.requestUri != null ? this.requestUri : this.delegate.get().getRequestURI());
        }

        public String getContextPath() {
            return this.forwardedPrefix != null ? this.forwardedPrefix : this.delegate.get().getContextPath();
        }

        public String getRequestUri() {
            if (this.requestUri == null) {
                return this.delegate.get().getRequestURI();
            }
            this.recalculatePathsIfNecessary();
            return this.requestUri;
        }

        public StringBuffer getRequestUrl() {
            this.recalculatePathsIfNecessary();
            return new StringBuffer(this.requestUrl);
        }

        private void recalculatePathsIfNecessary() {
            if (!this.actualRequestUri.equals(this.delegate.get().getRequestURI())) {
                this.actualRequestUri = this.delegate.get().getRequestURI();
                this.requestUri = this.initRequestUri();
                this.requestUrl = this.initRequestUrl();
            }
        }

        @Nullable
        public String getErrorRequestUri() {
            HttpServletRequest request = this.delegate.get();
            String requestUri = (String)request.getAttribute("jakarta.servlet.error.request_uri");
            if (this.forwardedPrefix == null || requestUri == null) {
                return requestUri;
            }
            ErrorPathRequest errorRequest = new ErrorPathRequest((ServletRequest)request);
            return this.forwardedPrefix + UrlPathHelper.rawPathInstance.getPathWithinApplication(errorRequest);
        }
    }
}

