/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.format.datetime.standard;

import java.time.Duration;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.springframework.format.annotation.DurationFormat;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

public abstract class DurationFormatterUtils {
    private static final Pattern ISO_8601_PATTERN = Pattern.compile("^[+-]?[pP].*$");
    private static final Pattern SIMPLE_PATTERN = Pattern.compile("^([+-]?\\d+)([a-zA-Z]{0,2})$");
    private static final Pattern COMPOSITE_PATTERN = Pattern.compile("^([+-]?)\\(?\\s?(\\d+d)?\\s?(\\d+h)?\\s?(\\d+m)?\\s?(\\d+s)?\\s?(\\d+ms)?\\s?(\\d+us)?\\s?(\\d+ns)?\\)?$");

    private DurationFormatterUtils() {
    }

    public static Duration parse(String value, DurationFormat.Style style) {
        return DurationFormatterUtils.parse(value, style, null);
    }

    public static Duration parse(String value, DurationFormat.Style style, @Nullable DurationFormat.Unit unit) {
        Assert.hasText(value, () -> "Value must not be empty");
        return switch (style) {
            default -> throw new IncompatibleClassChangeError();
            case DurationFormat.Style.ISO8601 -> DurationFormatterUtils.parseIso8601(value);
            case DurationFormat.Style.SIMPLE -> DurationFormatterUtils.parseSimple(value, unit);
            case DurationFormat.Style.COMPOSITE -> DurationFormatterUtils.parseComposite(value);
        };
    }

    public static String print(Duration value, DurationFormat.Style style) {
        return DurationFormatterUtils.print(value, style, null);
    }

    public static String print(Duration value, DurationFormat.Style style, @Nullable DurationFormat.Unit unit) {
        return switch (style) {
            default -> throw new IncompatibleClassChangeError();
            case DurationFormat.Style.ISO8601 -> value.toString();
            case DurationFormat.Style.SIMPLE -> DurationFormatterUtils.printSimple(value, unit);
            case DurationFormat.Style.COMPOSITE -> DurationFormatterUtils.printComposite(value);
        };
    }

    public static Duration detectAndParse(String value) {
        return DurationFormatterUtils.detectAndParse(value, null);
    }

    public static Duration detectAndParse(String value, @Nullable DurationFormat.Unit unit) {
        return DurationFormatterUtils.parse(value, DurationFormatterUtils.detect(value), unit);
    }

    public static DurationFormat.Style detect(String value) {
        Assert.notNull((Object)value, "Value must not be null");
        if (ISO_8601_PATTERN.matcher(value).matches()) {
            return DurationFormat.Style.ISO8601;
        }
        if (SIMPLE_PATTERN.matcher(value).matches()) {
            return DurationFormat.Style.SIMPLE;
        }
        if (COMPOSITE_PATTERN.matcher(value).matches()) {
            return DurationFormat.Style.COMPOSITE;
        }
        throw new IllegalArgumentException("'" + value + "' is not a valid duration, cannot detect any known style");
    }

    private static Duration parseIso8601(String value) {
        try {
            return Duration.parse(value);
        }
        catch (Exception ex) {
            throw new IllegalArgumentException("'" + value + "' is not a valid ISO-8601 duration", ex);
        }
    }

    private static Duration parseSimple(String text, @Nullable DurationFormat.Unit fallbackUnit) {
        try {
            DurationFormat.Unit parsingUnit;
            Matcher matcher = SIMPLE_PATTERN.matcher(text);
            Assert.state(matcher.matches(), "Does not match simple duration pattern");
            String suffix = matcher.group(2);
            DurationFormat.Unit unit = parsingUnit = fallbackUnit == null ? DurationFormat.Unit.MILLIS : fallbackUnit;
            if (StringUtils.hasLength(suffix)) {
                parsingUnit = DurationFormat.Unit.fromSuffix(suffix);
            }
            return parsingUnit.parse(matcher.group(1));
        }
        catch (Exception ex) {
            throw new IllegalArgumentException("'" + text + "' is not a valid simple duration", ex);
        }
    }

    private static String printSimple(Duration duration, @Nullable DurationFormat.Unit unit) {
        unit = unit == null ? DurationFormat.Unit.MILLIS : unit;
        return unit.print(duration);
    }

    private static Duration parseComposite(String text) {
        try {
            Matcher matcher = COMPOSITE_PATTERN.matcher(text);
            Assert.state(matcher.matches() && matcher.groupCount() > 1, "Does not match composite duration pattern");
            String sign = matcher.group(1);
            boolean negative = sign != null && sign.equals("-");
            Duration result = Duration.ZERO;
            DurationFormat.Unit[] units = DurationFormat.Unit.values();
            for (int i2 = 2; i2 < matcher.groupCount() + 1; ++i2) {
                String segment = matcher.group(i2);
                if (!StringUtils.hasText(segment)) continue;
                DurationFormat.Unit unit = units[units.length - i2 + 1];
                result = result.plus(unit.parse(segment.replace(unit.asSuffix(), "")));
            }
            return negative ? result.negated() : result;
        }
        catch (Exception ex) {
            throw new IllegalArgumentException("'" + text + "' is not a valid composite duration", ex);
        }
    }

    private static String printComposite(Duration duration) {
        long nanos;
        int millis;
        int seconds;
        int minutes;
        int hours;
        long days;
        if (duration.isZero()) {
            return DurationFormat.Unit.SECONDS.print(duration);
        }
        StringBuilder result = new StringBuilder();
        if (duration.isNegative()) {
            result.append('-');
            duration = duration.negated();
        }
        if ((days = duration.toDaysPart()) != 0L) {
            result.append(days).append(DurationFormat.Unit.DAYS.asSuffix());
        }
        if ((hours = duration.toHoursPart()) != 0) {
            result.append(hours).append(DurationFormat.Unit.HOURS.asSuffix());
        }
        if ((minutes = duration.toMinutesPart()) != 0) {
            result.append(minutes).append(DurationFormat.Unit.MINUTES.asSuffix());
        }
        if ((seconds = duration.toSecondsPart()) != 0) {
            result.append(seconds).append(DurationFormat.Unit.SECONDS.asSuffix());
        }
        if ((millis = duration.toMillisPart()) != 0) {
            result.append(millis).append(DurationFormat.Unit.MILLIS.asSuffix());
        }
        if ((nanos = (long)duration.toNanosPart() - Duration.ofMillis(millis).toNanos()) != 0L) {
            long micros = nanos / 1000L;
            long remainder = nanos - micros * 1000L;
            if (micros > 0L) {
                result.append(micros).append(DurationFormat.Unit.MICROS.asSuffix());
            }
            if (remainder > 0L) {
                result.append(remainder).append(DurationFormat.Unit.NANOS.asSuffix());
            }
        }
        return result.toString();
    }
}

